/*  Erzeugt zusätzliche Blockierungen für Überlappungen des gegeben AGs mit anderen AGen auf der Hauptressource. Dafür wird
    zuerst die Dauer der Überlappungen mit anderen AGen mittels der Funktion
    scheduling.resource_timeline__overlap_duration__by__a2_id() ermittelt und dann die Funktion
    scheduling.resource_timeline__ab2__block_duration() mit diesem Wert als Parameter gerufen.

    Dabei werden bei einem gegebenen AG nur die Überlappungen mit AGen gezählt, welche früher beendet sind oder bei
    gleichzeitigem Ende mit dem gegeben AG, welche eine kleinere a2_id haben.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__ab2__block_overlap', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__ab2__block_overlap(
    _ab2                    ab2,
    _blocktime_date_start   timestamp = null,    -- Startzeitpunkt, ab dem Blockierungen eingetragen werden sollen.
    _loglevel               int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
) RETURNS bool AS $$
DECLARE
    _prefix             varchar := 'scheduling.resource_timeline__ab2__block_overlap -';

    _overlap_duration   numeric;

BEGIN
    -- Debug
    IF _loglevel >= 4 THEN
        RAISE NOTICE '% a2_id:%;', _prefix, _ab2.a2_id;
    END IF;

    -- not found, do nothing
    IF NOT EXISTS (SELECT true FROM ab2 WHERE a2_id = _ab2.a2_id ) THEN
        RAISE NOTICE 'AB2 % not found', _ab2.a2_id;
        RETURN false;
    END IF;

    -- not scheduled, do nothing
    IF NOT EXISTS ( SELECT true FROM scheduling.resource_timeline WHERE ti_a2_id = _ab2.a2_id AND ti_type IN ( 'task', 'task.buffer' ) ) THEN
        RAISE NOTICE 'AB2 % not scheduled', _ab2.a2_id;
        RETURN false;
    END IF;

    -- Dauer der Überlappungen auf der Hauptressource mit anderen AGen ermitteln.
    _overlap_duration :=  scheduling.resource_timeline__overlap_duration__by__a2_id(
                              _a2_id                => _ab2.a2_id
                            , _filter_for_other_ab2 => 'dlz_a2_et'
                          );

    -- Debug
    IF _loglevel >= 5 THEN
        RAISE NOTICE '% overlap:%, _overlap_duration, %;', _prefix, make_interval( secs => _overlap_duration ), _overlap_duration;
    END IF;

    -- Errechnete Dauer der Überlappung als Überhang blockieren.
    RETURN scheduling.resource_timeline__ab2__block_duration(
                _ab2_id               => _ab2.a2_id
              , _blocktime_duration   => _overlap_duration
              , _blocktime_date_start => coalesce( _blocktime_date_start, _ab2.a2_et )
              , _loglevel             => _loglevel
            );

END $$ LANGUAGE plpgsql;